<?php
function clean_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}

function handle_file_upload($file) {
    $upload_dir = UPLOAD_PATH;
    
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx'];
    
    if (!in_array($file_extension, $allowed_extensions)) {
        throw new Exception('Invalid file type. Allowed: ' . implode(', ', $allowed_extensions));
    }
    
    if ($file['size'] > MAX_FILE_SIZE) {
        throw new Exception('File size too large. Maximum: ' . (MAX_FILE_SIZE / 1024 / 1024) . 'MB');
    }
    
    $filename = uniqid() . '_' . time() . '.' . $file_extension;
    $filepath = $upload_dir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return $filename;
    }
    
    throw new Exception('Failed to upload file.');
}

function save_customer($db, $data) {
    // Check if customer already exists
    $existing = $db->query(
        "SELECT id FROM customers WHERE email = ?",
        [$data['email']]
    )->fetch();
    
    if ($existing) {
        return $existing['id'];
    }
    
    // Insert new customer
    $db->query(
        "INSERT INTO customers (customer_name, email, phone, company) VALUES (?, ?, ?, ?)",
        [$data['customer_name'], $data['email'], $data['phone'], $data['company']]
    );
    
    return $db->lastInsertId();
}

function save_inquiry($db, $data) {
    $db->query(
        "INSERT INTO product_inquiries (customer_id, product_name, product_description, quantity, specifications, reference_photos) 
         VALUES (?, ?, ?, ?, ?, ?)",
        [$data['customer_id'], $data['product_name'], $data['product_description'], $data['quantity'], $data['specifications'], $data['reference_photos']]
    );
    
    return $db->lastInsertId();
}

function get_dashboard_stats($db) {
    $stats = [];
    
    // Total inquiries
    $result = $db->query("SELECT COUNT(*) as count FROM product_inquiries")->fetch();
    $stats['total_inquiries'] = $result['count'];
    
    // Pending inquiries
    $result = $db->query("SELECT COUNT(*) as count FROM product_inquiries WHERE status = 'pending'")->fetch();
    $stats['pending_inquiries'] = $result['count'];
    
    // Total PIs
    $result = $db->query("SELECT COUNT(*) as count FROM proforma_invoices")->fetch();
    $stats['total_pis'] = $result['count'];
    
    // Total customers
    $result = $db->query("SELECT COUNT(*) as count FROM customers")->fetch();
    $stats['total_customers'] = $result['count'];
    
    return $stats;
}

function get_recent_inquiries($db, $limit = 5) {
    return $db->query(
        "SELECT i.*, c.customer_name 
         FROM product_inquiries i 
         LEFT JOIN customers c ON i.customer_id = c.id 
         ORDER BY i.created_at DESC 
         LIMIT ?",
        [$limit]
    )->fetchAll();
}

function create_proforma_invoice($db, $data) {
    // Generate PI number
    $pi_number = PI_PREFIX . date('Ymd') . '-' . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT);
    
    $db->query(
        "INSERT INTO proforma_invoices (pi_number, customer_id, inquiry_id, items, subtotal, tax_amount, total_amount, currency, validity_date, terms_conditions, created_by) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
        [$pi_number, $data['customer_id'], $data['inquiry_id'], $data['items'], $data['subtotal'], $data['tax_amount'], $data['total_amount'], $data['currency'], $data['validity_date'], $data['terms_conditions'], $_SESSION['user_id']]
    );
    
    return $db->lastInsertId();
}

function get_customer_history($db, $customer_id) {
    return $db->query(
        "SELECT pi.*, i.product_name, i.quantity as inquiry_quantity
         FROM proforma_invoices pi
         LEFT JOIN product_inquiries i ON pi.inquiry_id = i.id
         WHERE pi.customer_id = ?
         ORDER BY pi.created_at DESC",
        [$customer_id]
    )->fetchAll();
}
?>